document.addEventListener("DOMContentLoaded", function () {
  const chatBox = document.getElementById("chat-box");
  const nameInput = document.getElementById("name");
  const companyInput = document.getElementById("company");
  const employeesInput = document.getElementById("employees");
  const phoneInput = document.getElementById("phone");
  const messageInput = document.getElementById("message");

  const sendNameButton = document.getElementById("send-name");
  const sendCompanyButton = document.getElementById("send-company");
  const sendEmployeesButton = document.getElementById("send-employees");
  const sendPhoneButton = document.getElementById("send-phone");
  const sendMessageButton = document.getElementById("send-message");
  const phoneError = document.getElementById("phone-error");

  const chatForm = document.getElementById("chat-form");

  let formSubmitted = false; // Flag para verificar se o formulário já foi enviado

  // Função para validar o telefone
  function validatePhone(phone) {
      const phonePattern = /^\(\d{2}\) \d{5}-\d{4}$/;
      return phonePattern.test(phone);
  }

  // Função para formatar o telefone
  function formatPhone(input) {
      let value = input.value.replace(/\D/g, ""); // Remove todos os caracteres não numéricos
      if (value.length > 2) {
          value = `(${value.substring(0, 2)}) ${value.substring(2, 7)}-${value.substring(7, 11)}`;
      }
      input.value = value;
  }

  // Função para adicionar mensagens ao chat
  function addMessageToChat(message, isUser) {
      const messageDiv = document.createElement("div");
      messageDiv.classList.add("chat-message", isUser ? "user" : "bot");
      messageDiv.textContent = message;
      chatBox.appendChild(messageDiv);
      chatBox.scrollTop = chatBox.scrollHeight; // Rola para a última mensagem
  }

  // Função para avançar para o próximo campo
  function advanceToNextField(currentField, nextField, botMessage) {
      if (currentField.value.trim() !== "") {
          addMessageToChat(currentField.value.trim(), true);
          addMessageToChat(botMessage, false);
          currentField.parentElement.classList.add("hidden");
          currentField.nextElementSibling.classList.add("hidden"); // Oculta o botão atual
          if (nextField) {
              nextField.parentElement.classList.remove("hidden");
              nextField.focus();
          } else {
              // Quando o último campo é preenchido, envia os dados automaticamente
              sendFormData();
          }
      }
  }

  // Função para enviar os dados do formulário
  function sendFormData() {
      if (formSubmitted) {
          addMessageToChat("Você já enviou suas informações. Entraremos em contato em breve!", false);
          return;
      }

      const formData = {
          name: nameInput.value.trim(),
          company: companyInput.value.trim(),
          employees: employeesInput.value.trim(),
          phone: phoneInput.value.trim(),
          message: messageInput.value.trim(),
      };

      // Enviar os dados via AJAX
      fetch("send_email.php", {
          method: "POST",
          headers: {
              "Content-Type": "application/json",
          },
          body: JSON.stringify(formData),
      })
          .then((response) => response.json())
          .then((data) => {
              if (data.status === "success") {
                  addMessageToChat("Em breve entraremos em contato com você!", false);
                  formSubmitted = true; // Marca o formulário como enviado
                  disableForm(); // Desabilita o formulário após o envio
              } else {
                  addMessageToChat("Ocorreu um erro ao enviar seus dados. Tente novamente.", false);
              }
          })
          .catch((error) => {
              console.error("Erro:", error);
              addMessageToChat("Ocorreu um erro ao enviar seus dados. Tente novamente.", false);
          });

      // Reseta o formulário após o envio
      resetForm();
  }

  // Função para desabilitar o formulário após o envio
  function disableForm() {
      nameInput.disabled = true;
      companyInput.disabled = true;
      employeesInput.disabled = true;
      phoneInput.disabled = true;
      messageInput.disabled = true;
      sendNameButton.disabled = true;
      sendCompanyButton.disabled = true;
      sendEmployeesButton.disabled = true;
      sendPhoneButton.disabled = true;
      sendMessageButton.disabled = true;

      // Oculta o campo de mensagem e o botão de envio
      messageInput.parentElement.classList.add("hidden");
      sendMessageButton.classList.add("hidden");

      // Exibe uma mensagem final
      addMessageToChat("Obrigado pelas informações! Entraremos em contato em breve.", false);
  }

  // Função para resetar o formulário
  function resetForm() {
      nameInput.value = "";
      companyInput.value = "";
      employeesInput.value = "";
      phoneInput.value = "";
      messageInput.value = "";
      nameInput.parentElement.classList.remove("hidden");
      sendNameButton.classList.remove("hidden");
      companyInput.parentElement.classList.add("hidden");
      sendCompanyButton.classList.add("hidden");
      employeesInput.parentElement.classList.add("hidden");
      sendEmployeesButton.classList.add("hidden");
      phoneInput.parentElement.classList.add("hidden");
      sendPhoneButton.classList.add("hidden");
      messageInput.parentElement.classList.add("hidden");
      sendMessageButton.classList.add("hidden");
  }

  // Função para ajustar a altura da área de mensagens
  function adjustChatBoxHeight() {
      const isMobile = window.innerWidth <= 480; // Verifica se é um dispositivo móvel
      if (isMobile) {
          chatBox.style.height = "50vh"; // Define a altura do chat como 50% da tela
      } else {
          chatBox.style.height = "auto"; // Restaura a altura padrão em desktop
      }
  }

  // Ajusta a altura inicial
  adjustChatBoxHeight();

  // Observa mudanças no tamanho da janela (útil para quando o teclado é aberto/fechado)
  window.addEventListener("resize", adjustChatBoxHeight);

  // Foca no campo de entrada quando o formulário é clicado
  chatForm.addEventListener("click", function () {
      const activeInput = chatForm.querySelector("input:not(.hidden), textarea:not(.hidden)");
      if (activeInput) {
          activeInput.focus();
      }
  });

  // Event listeners para os botões de envio
  sendNameButton.addEventListener("click", () => {
      advanceToNextField(nameInput, companyInput, "Obrigado! Qual é o nome da sua empresa?");
  });

  sendCompanyButton.addEventListener("click", () => {
      advanceToNextField(companyInput, employeesInput, "Quantos funcionários sua empresa tem?");
  });

  sendEmployeesButton.addEventListener("click", () => {
      advanceToNextField(employeesInput, phoneInput, "Agora, qual é o seu telefone?");
  });

  sendPhoneButton.addEventListener("click", () => {
      if (validatePhone(phoneInput.value.trim())) {
          phoneError.style.display = "none";
          advanceToNextField(phoneInput, messageInput, "Por fim, digite sua mensagem.");
      } else {
          phoneError.style.display = "block";
      }
  });

  sendMessageButton.addEventListener("click", () => {
      if (messageInput.value.trim() !== "") {
          advanceToNextField(messageInput, null, "Obrigado pelas informações!");
      }
  });

  // Event listeners para a tecla Enter
  const handleEnterKey = (event, currentField, nextField, botMessage) => {
      if (event.key === "Enter") {
          event.preventDefault();
          if (currentField.value.trim() !== "") {
              advanceToNextField(currentField, nextField, botMessage);
          }
      }
  };

  nameInput.addEventListener("keypress", (event) => {
      handleEnterKey(event, nameInput, companyInput, "Obrigado! Qual é o nome da sua empresa?");
  });

  companyInput.addEventListener("keypress", (event) => {
      handleEnterKey(event, companyInput, employeesInput, "Quantos funcionários sua empresa tem?");
  });

  employeesInput.addEventListener("keypress", (event) => {
      if (event.key === "Enter") {
          event.preventDefault();
          if (employeesInput.value.trim() !== "") {
              advanceToNextField(employeesInput, phoneInput, "Agora, qual é o seu telefone?");
          }
      }
  });

  phoneInput.addEventListener("keypress", (event) => {
      if (event.key === "Enter") {
          event.preventDefault();
          if (validatePhone(phoneInput.value.trim())) {
              phoneError.style.display = "none";
              advanceToNextField(phoneInput, messageInput, "Por fim, digite sua mensagem.");
          } else {
              phoneError.style.display = "block";
          }
      }
  });

  messageInput.addEventListener("keypress", (event) => {
      if (event.key === "Enter") {
          event.preventDefault();
          if (messageInput.value.trim() !== "") {
              advanceToNextField(messageInput, null, "Obrigado pelas informações!");
          }
      }
  });

  // Formatação do telefone enquanto digita
  phoneInput.addEventListener("input", () => {
      if (validatePhone(phoneInput.value.trim())) {
          phoneError.style.display = "none";
      }
      formatPhone(phoneInput);
  });

  // Permitir apagar caracteres no telefone
  phoneInput.addEventListener("keydown", (event) => {
      if (event.key === "Backspace" || event.key === "Delete") {
          setTimeout(() => {
              formatPhone(phoneInput);
          }, 0);
      }
  });
});